// Copyright 2014 Google Inc. All Rights Reserved.

#ifndef ANDROID_AUTO_PROJECTION_PROTOCOL_RADIO_ENDPOINT_H
#define ANDROID_AUTO_PROJECTION_PROTOCOL_RADIO_ENDPOINT_H

#include "common.h"
#include "IRadioCallbacks.h"
#include "ProtocolEndpointBase.h"

/**
 * Use this class to enable control of the car's AM/FM radio tuner from the phone. These are
 * the steps necessary for an integration. You must also subclass IRadioCallbacks and implement
 * all the methods so that the GAL receiver can interface with your hardware. The call sequence
 * would look something like this.
 * <br>
 * <pre>
 *      galReceiver->init();
 *      ... Initialization code ...
 *      RadioEndpoint* endpoint = new RadioEndpoint(serviceId, galReceiver->messageRouter());
 *      endpoint->registerCallbacks(callbacks); // Subclassed from IRadioCallbacks.
 *      endpoint->setRadioConfiguration(...);
 *      galReceiver->registerService(endpoint);
 *      ... Other Initialization code ...
 *      galReceiver->start();
 * </pre>
 */
class RadioEndpoint : public ProtocolEndpointBase {
public:
    RadioEndpoint(uint8_t id, MessageRouter* router) : ProtocolEndpointBase(id, router, false) {}

    void addDiscoveryInfo(ServiceDiscoveryResponse *sdr);
    int routeMessage(uint8_t channelId, uint16_t type, const shared_ptr<IoBuffer>& msg);
    void onChannelOpened(uint8_t channelId);

    /**
     * Call this method during initialization to set up radio.
     * @param config radio configuration to set.
     */
    void setRadioConfiguration(const RadioConfigurationStruct &config);
    /**
     * You must call this prior to initialization with a properly subclassed IRadioCallbacks.
     * @param radioCallbacks The callbacks that should be invoked.
     */
    void registerCallbacks(const shared_ptr<IRadioCallbacks>& radioCallbacks);
    /**
     * Call this when the radio source has changed (via MD or user making the
     * change via native mode).
     * @param status if this is a response to a RadioSourceRequest, this
     * should be set to STATUS_SUCCESS or an error code (see enum MESSAGE_STATUS),
     * or STATUS_UNSOLICITED_MESSAGE if this was caused by the user in native
     * mode.
     * @param enabled if set, radio is the selected source.
     */
    int sendRadioSourceResponse(int32_t status, bool enabled);
    /**
     * Call this method in response to a StepChannelRequest (or if the user
     * initiated a step via the native mode), the tuning is done
     * asynchronously, which will result in a RadioStationInfoNotification.
     * @param status STATUS_SUCCESS if host was able to step to the next channel
     * (see enum MESSAGE_STATUS) or STATUS_UNSOLICITED_MESSAGE if this was
     * caused by the user in native mode.
     * @param radioId The ID of the radio that handled the step request.
     */
    int sendStepChannelResponse(int32_t status, int32_t radioId);
    /**
     * Call this method in response to a TuneToStationRequest (or if the user
     * initiated a tune via the native mode, i.e. station preset), the tuning is done
     * asynchronously, which will result in a RadioStationInfoNotification.
     * @param status STATUS_SUCCESS if host was able to tune to specified station
     * (see enum MESSAGE_STATUS) or STATUS_UNSOLICITED_MESSAGE if this was caused
     * by the user in native mode.
     * @param radioId The ID of the radio that handled the tune request.
     * This parameter is ignored if status != STATUS_SUCCESS.
     */
    int sendTuneToStationResponse(int32_t status, int32_t radioId);
    /**
     * Call this method in response to a SeekStationRequest (or if the user
     * initiated a seek via the native mode), the tuning is done
     * asynchronously, which will result in a RadioStationInfoNotification.
     * @param status STATUS_SUCCESS if host has completed seek
     * (see enum MESSAGE_STATUS) or STATUS_UNSOLICITED_MESSAGE if this was
     * caused by the user in native mode.
     * @param radioId The ID of the radio that handled the step request.
     * This parameter is ignored if status != STATUS_SUCCESS.
     */
    int sendSeekStationResponse(int32_t status, int32_t radioId);
    /**
     * Call this method in response to a ScanStationsRequest (or if the user
     * initiated a scan via the native mode), the tuning is done
     * asynchronously, which will result in a RadioStationInfoNotification
     * message for each scanned station.
     * @param status STATUS_SUCCESS if host has successfully processed the
     * scan start/stop command (see enum MESSAGE_STATUS) or
     * STATUS_UNSOLICITED_MESSAGE if this was caused by the user in native
     * mode.
     * @param radioId The ID of the radio that handled the scan request.
     * @param started Flag indicating if scan started or stopped.  Ignored if
     * status != STATUS_SUCCESS.
     */
    int sendScanStationsResponse(int32_t status, int32_t radioId, bool started);
    /**
     * Call this method when new station info is available to send to the client.
     * @param status STATUS_SUCCESS if host is able to retrieve station info.
     * (see enum MESSAGE_STATUS).
     * @param stationInfo station info to send.
     * @return STATUS_SUCCESS if the call succeeded, an error code otherwise.
     */
    int sendRadioStationInfoNotification(int32_t radioId,
                                         const StationInfoStruct& stationInfo);
    /**
     * Call this method to send the active radio to the client.
     * @param status STATUS_SUCCESS if there's no error
     * (see enum MESSAGE_STATUS) or STATUS_UNSOLICITED_MESSAGE if this was
     * caused by the user in native mode.
     * @param radioId ID of the active radio.
     * @param stationInfo station info to send.
     * @return STATUS_SUCCESS if the call succeeded, an error code otherwise.
     */
    int sendActiveRadioNotification(int32_t status, int32_t radioId,
                                    const StationInfoStruct& stationInfo);
    /**
     * Call this method to send the station presets to the client.
     * @param presets station presets stored on the host.
     * @return STATUS_SUCCESS if the call succeeded, an error code otherwise.
     */
    int sendStationPresetsNotification(const vector<StationPresetListStruct> &presetLists);

    /**
     * Call this method when there's new traffic update to be sent to the client.
     * @param status STATUS_SUCCESS if there's no error
     * (see enum MESSAGE_STATUS).
     * @param radioId ID of radio sending the response.
     * @param incidents all the available traffic incidents.
     * @return STATUS_SUCCESS if the call succeeded, an error code otherwise.
     */
    int sendGetTrafficUpdateResponse(int32_t status, int32_t radioId,
                                     const vector<TrafficIncidentStruct> &incidents);
    /**
     * Call this method in response to a MuteRadioRequest.
     * @param status STATUS_SUCCESS if there's no error
     * (see enum MESSAGE_STATUS) or STATUS_UNSOLICITED_MESSAGE if this was
     * caused by the user in native mode.
     * @param radioId id of radio
     * @param muted whether radio is muted or not
     * @return STATUS_SUCCESS if the call succeeded, an error code otherwise.
     */
    int sendMuteRadioResponse(int32_t status, int32_t radioId, bool muted);
    /**
     * Call this method in response to a CancelRadioOperationsRequest.
     * @param status STATUS_SUCCESS if there's no error
     * (see enum MESSAGE_STATUS).
     * @param radioId ID or radio this status pertains to.
     * @return STATUS_SUCCESS if the call succeeded, an error code otherwise.
     */
    int sendCancelRadioOperationsResponse(int32_t status, int32_t radioId);
    /**
     * Call this method in response to a ConfigureChannelSpacingRequest.
     * @param status STATUS_SUCCESS if there's no error
     * (see enum MESSAGE_STATUS) or STATUS_UNSOLICITED_MESSAGE if this was
     * caused by the user in native mode.
     * @param radioId ID or radio this configuration pertains to.
     * @param channelSpacing channel spacing value the radio is configured to.
     * @return STATUS_SUCCESS if the call succeeded, an error code otherwise.
     */
    int sendConfigureChannelSpacingResponse(int32_t status, int32_t radioId,
                                            int32_t channelSpacing);
    /**
     * Call this method in response to a GetProgramListRequest.
     * Since this operation can be time consuming, the host may call this
     * multiple times to "stream" the result back to the client as new programs
     * are discovered.
     * @param status STATUS_SUCCESS if there's no error
     * (see enum MESSAGE_STATUS).
     * @param radioId ID of radio that's doing the program list retrieval.
     * @param isLastResponse set to true if this is the lst message in response
     * to the request (done scanning for programs).
     * @param stationInfos list of station infos of the newly discovered
     * programs (if the response is broken into multiple messages, this list
     * should only contain the new programs discovered since the previous message
     * was sent).
     * @return STATUS_SUCCESS if the call succeeded, an error code otherwise.
     */
    int sendGetProgramListResponse(int32_t status, int32_t radioId,
                                   bool isLast,
                                   const vector<StationInfoStruct> &stationInfos);
private:
    void populateStationInfo(const StationInfoStruct &stationInfo, RadioStationInfo *proto);
    void populateMetaData(const StationMetaDataStruct &metaData, RadioStationMetaData *proto);
    void populateRdsData(const RdsDataStruct& rdsData, RdsData* proto);
    void populateHdRadioData(const HdRadioDataStruct& hdData, HdRadioStationInfo* proto);

    int handleStepRequest(const StepChannelRequest &req);
    int handleSeekRequest(const SeekStationRequest &req);
    int handleScanRequest(const ScanStationsRequest &req);
    int handleTuneRequest(const TuneToStationRequest &req);
    int handleGetProgramListRequest(const GetProgramListRequest &req);
    int handleCancelRequest(const CancelRadioOperationsRequest &req);
    int handleConfigChannelSpacingRequest(const ConfigureChannelSpacingRequest &req);
    int handleGetTrafficUpdateRequest(const GetTrafficUpdateRequest &req);
    int handleOnMuteRadioRequest(const MuteRadioRequest &req);
    int handleRadioSourceRequest(const RadioSourceRequest &req);
    int handleSelectActiveRadioRequest(const SelectActiveRadioRequest &req);

    RadioConfigurationStruct mRadioConfig;
    shared_ptr<IRadioCallbacks> mRadioCallbacks;
};
#endif
